//
// Copyright 2011 Justin.tv. All rights reserved.
//

#import "TWAnalyticsController.h"
#import "Mixpanel.h"

static NSString *const TWTwitchAnalyticsMixpanelToken = @"96c97b9b4fa9f08a8ceef01d2abffd93";

static NSString *const TWAnalyticsMixpanelEventsDefaultsKey = @"mixpanel_distinct_id_events";
static NSString *const TWAnalyticsMixpanelDistinctIdDefaultsKey = @"mixpanel_distinct_id";
static NSString *const TWAnalyticsSpadeDeviceIdDefaultsKey = @"spade_device_id";
static NSString *const TWAnalyticsMixpanelBornVersionDefaultsKey = @"mixpanel_born_version";
static NSString *const TWAnalyticsFeatureBucketDefaultsKey = @"mixpanel_feature_bucket";

static NSString *const TWAnalyticsEventAPIErrorOccurred = @"api-error";
static NSString *const TWAnalyticsMixpanelPropertyDistinctId = @"distinct_id";
static NSString *const TWAnalyticsMixpanelPropertySource = @"source";
static NSString *const TWAnalyticsMixpanelPropertyPlayer = @"player";
static NSString *const TWAnalyticsMixpanelPropertyBornVersion = @"$born-version";
static NSString *const TWAnalyticsMixpanelPropertyFeatureBucket = @"feature_bucket";
static NSString *const TWAnalyticsSpadePropertyDeviceId = @"device_id";
static NSString *const TWAnalyticsPropertyLogin = @"login";
static NSString *const TWAnalyticsPropertyDeviceOrientation = @"orientation";
static NSString *const TWAnalyticsPropertyLoggedIn = @"logged_in";
static NSString *const TWAnalyticsPropertyDirectory = @"directory";
static NSString *const TWAnalyticsPropertyViewable = @"viewable";
static NSString *const TWAnalyticsPropertyPlatform = @"platform";
static NSString *const TWAnalyticsPropertyDeviceModel = @"device_model";
static NSString *const TWAnalyticsPropertySystemVersion = @"system_version";
static NSString *const TWAnalyticsPropertyAppVersion = @"app_version";
static NSString *const TWAnalyticsPropertyAppBuild = @"app_build";
static NSString *const TWanalyticsPropertyAppBundle = @"bundle_identifier";
static NSString *const TWAnalyticsPropertyMinutesLogged = @"minutes_logged";

NSString *const TWAnalyticsPropertyReferrer = @"referrer";
NSString *const TWAnalyticsPropertyType = @"type";
NSString *const TWAnalyticsPropertyAction = @"action";
NSString *const TWAnalyticsPropertyLocation = @"location";
NSString *const TWAnalyticsPropertyGame = @"game";
NSString *const TWAnalyticsPropertyChannel = @"channel";

NSString *const TWAnalyticsEventErrorOccurred = @"error-occurred";
NSString *const TWAnalyticsEventStreamsBrowse = @"plugin-streams-browse";
NSString *const TWAnalyticsEventStreamPlay = @"plugin-stream-play";
NSString *const TWAnalyticsEventAppInstallPrompt = @"plugin-app-install";

NSString *const TWAnalyticsReferrerChannels = @"channels";

@interface TWAnalyticsController () <MixpanelDelegate>
@end

@interface TWAnalyticsController (Private)
- (void) _trackEvent:(NSString *) event withDetails:(NSMutableDictionary *) details;
@end

@implementation TWAnalyticsController {
	
	NSString *_mixpanelDistinctIdentifier;
	NSString *_spadeDeviceIdentifier;
	
	BOOL _shouldFlushMixpanelProfileEvents;
}

+ (TWAnalyticsController *) analyticsController {
	static TWAnalyticsController *analyticsController = nil;
	
	tw_dispatch_once(^{
		analyticsController = [[[self class] alloc] init];
	});
	
	return analyticsController;
}

- (void)dealloc
{
	[NSObject cancelPreviousPerformRequestsWithTarget:self];
	[[NSNotificationCenter defaultCenter] removeObserver:self];
}

- (NSString *)analyticsIdentifier
{
	return _spadeDeviceIdentifier;
}

- (NSString *)randomDistinctId
{
	// NSUUID generates a random 128-bit UUID which is printed as hyphen-punctuated ASCII
	// ex: 68753A44-4D6F-1226-9C60-0050E4C00067
	
	NSUUID *uuid = [[NSUUID alloc] init];
	return [[uuid UUIDString] tw_stringByRemovingCharactersInSet:[NSCharacterSet characterSetWithCharactersInString:@"-"]];
}

- (id) init {
	if (!(self = [super init]))
		return nil;
	
	// Initialize MixPanel
	Mixpanel *mixpanel = [Mixpanel sharedInstanceWithToken:TWTwitchAnalyticsMixpanelToken];
	mixpanel.delegate = self;
	mixpanel.sendDeviceInfo = NO;
	mixpanel.showNetworkActivityIndicator = NO;
	
	// Create distinct identifiers if none exists
	NSCharacterSet *hyphenSet = [NSCharacterSet characterSetWithCharactersInString:@"-"];
	NSString *defaultIdentifier = [[[[UIDevice currentDevice] identifierForVendor] UUIDString] tw_stringByRemovingCharactersInSet:hyphenSet];
	
	_mixpanelDistinctIdentifier = [[NSUserDefaults standardUserDefaults] objectForKey:TWAnalyticsMixpanelDistinctIdDefaultsKey];
	if (!_mixpanelDistinctIdentifier) {
		
		_mixpanelDistinctIdentifier = defaultIdentifier;
		[[NSUserDefaults standardUserDefaults] setObject:_mixpanelDistinctIdentifier forKey:TWAnalyticsMixpanelDistinctIdDefaultsKey];
	}
	_spadeDeviceIdentifier = [[NSUserDefaults standardUserDefaults] objectForKey:TWAnalyticsSpadeDeviceIdDefaultsKey];
	if (!_spadeDeviceIdentifier) {
		
		_spadeDeviceIdentifier = defaultIdentifier;
		[[NSUserDefaults standardUserDefaults] setObject:_spadeDeviceIdentifier forKey:TWAnalyticsSpadeDeviceIdDefaultsKey];
	}
	
	NSString *platformString = [UIDevice currentDevice].isPhone ? @"iphone_t" : @"ipad_t";
	[mixpanel registerSuperProperties:@{TWAnalyticsMixpanelPropertySource:platformString,
										TWAnalyticsMixpanelPropertyPlayer:platformString,
										TWAnalyticsSpadePropertyDeviceId:_spadeDeviceIdentifier,
										TWAnalyticsPropertyPlatform:@"ios",
										TWAnalyticsPropertyDeviceModel:[UIDevice currentDevice].tw_platformString,
										TWAnalyticsPropertySystemVersion:[NSString stringWithFormat:@"%@ %@", [UIDevice currentDevice].systemName, [UIDevice currentDevice].systemVersion],
										TWAnalyticsPropertyAppBuild:[NSBundle mainBundle].infoDictionary[@"CFBundleVersion"],
										TWAnalyticsPropertyAppVersion:[NSBundle mainBundle].infoDictionary[@"CFBundleShortVersionString"],
										TWanalyticsPropertyAppBundle:[NSBundle mainBundle].infoDictionary[@"CFBundleIdentifier"]}];
	
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(spadeEventForwardingDidFinish:) name:TWSpadeEventsRequestDidFinishNotification object:nil];
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(spadeEventForwardingDidFail:) name:TWSpadeEventsRequestDidFailNotification object:nil];
	
	return self;
}


- (void)refresh
{
	[[NSUserDefaults standardUserDefaults] synchronize];
}

- (void) trackError:(NSString *) error withDetails:(NSDictionary *) details
{
	if (!error.length) return;
	
	NSMutableDictionary *mutableErrorDetails = [details mutableCopy];
	mutableErrorDetails[TWAnalyticsPropertyType] = error;
	
	[self trackEvent:TWAnalyticsEventErrorOccurred withDetails:mutableErrorDetails];
}

- (void) trackEvent:(NSString *) event {
	
	[self trackEvent:event withDetails:[NSDictionary dictionary]];
}

- (void) trackEvent:(NSString *) event withDetails:(NSDictionary *) details {
	
	TWLog(@"ANALYTICS: Tracking event: %@ with details: %@",event,[details description]);
	
	NSMutableDictionary *properties = [NSMutableDictionary dictionaryWithDictionary:details];
	
	[[Mixpanel sharedInstance] track:event properties:properties];
}

#pragma mark -

- (BOOL)mixpanelWillFlush:(Mixpanel *)mixpanel
{
	// NSLog(@"MIXPANEL: MixPanel flushing!");
	return YES;
}

- (void)mixpanel:(Mixpanel *)mixpanel didFlushEvents:(NSString *)events
{
	// NSLog(@"MIXPANEL: Did flush events: %@",events);

	TWHTTPRequest *spadeRequest = [TWAPIRequest forwardMixPanelEventsToSpade:events];
	[[TWOperationQueue operationQueue] addOperation:spadeRequest];
}

#pragma mark Spade Event Forwarding

- (void)spadeEventForwardingDidFinish:(NSNotification *)notification
{
	TWLog(@"ANALYTICS: Spade event forwarding succeeded.");
}

- (void)spadeEventForwardingDidFail:(NSNotification *)notification
{
	TWLog(@"ANALYTICS: Spade event forwarding failed!");
}

@end
